# -*- coding: utf-8 -*-

import json
import os
import time
import random
import base64
import hashlib
import requests
from Crypto.Cipher import AES
from Crypto.Util.Padding import pad

# pip install pycryptodome
# pip install requests


class GcOpenApi:
    """
    甘草医生开放平台基础Api类
    """
    _instance = None

    @classmethod
    def get_instance(cls, *args):
        """
        获取模块静态对象引用，如果不存在则创建新的实例。
        :return: 返回模块对象
        """
        if cls._instance is None:
            cls._instance = cls(*args)
        return cls._instance

    def __init__(self, *args):
        # 调用者客户端信息
        self.USER_AGENT = 'gancao-openapi-dev/0.0.1 (.....)'
        # 延迟静态绑定：对象引用缓存池
        self.env = os.environ.get('ENV', 'test')  # ENV为环境变量，也可从外部传入
        print(f"当前环境：{self.env}")
        # TODO: 根据你自己本地环境修改判断条件
        if self.env == 'dev':
            # 测试环境地址
            self._url = 'http://dev-gapis-base.igancao.com/oapi'  # API网关地址
            self._ak = 'OU022B11I095379K7'  # 账户 access key
            self._sk = '46cb91b0f20c3e19'  # secret key
        else:
            # 生产环境地址
            self._url = 'https://oapi.igancao.com'
            self._ak = ''  # TODO: 填写你的生产环境 access key
            self._sk = ''  # TODO: 填写你的生产环境 secret key

    def exec_api(self, s_pkg: str, s_cls: str, post_data: dict={}, is_debug: bool = False) -> dict:
        """
        发起请求
        :param s_pkg: 包路径
        :param s_cls: 接口名称
        :param post_data: 入参，参考接口文档传入字典
        :param is_debug 是否打印结果日志
        :return: 返回字典，包含接口返回结果
        """
        post_data['package'] = s_pkg
        post_data['class'] = s_cls
        i_timestamp = int(time.time())
        s_data = json.dumps(post_data, ensure_ascii=False, separators=(',', ':'))
        # print(s_data)
        s_noise = self.rand_str(8)
        s_signature = hashlib.sha1(f"{s_data}{i_timestamp}{s_noise}{self._sk}".encode('utf-8')).hexdigest()
        s_data_aes = self.encrypt(s_data, self._sk)
        headers = {
            'Connection': 'close',
            'Content-Type': 'application/json; charset=utf-8',
            'Content-length': str(len(s_data_aes)),
            'Cache-Control': 'no-cache',
            'AK': self._ak,
            'Signature': s_signature,
            'UTC-Timestamp': str(i_timestamp),
            'NOISE': s_noise,
            'Expect': '',
            'User-Agent': self.USER_AGENT
        }
        start_time = time.time()
        response = requests.post(self._url, headers=headers, data=s_data_aes)
        used_time = time.time() - start_time
        log = [
            f"Interface url: {self._url}",
            f"Transmission time(ms): {used_time:.4f}",
            "Header:",
            "\t" + "\n\t".join(f"{key}: {value}" for key, value in headers.items()),
            "Send data:", self.json_format(s_data),
            "Send data(AES):", "\t" + s_data_aes,
            '-' * 20,
        ]
        if response.status_code == 200:
            de_response = self.decrypt(response.text, self._sk)
            if is_debug:
                log += ["Receive data:", self.json_format(de_response),  "Receive data(AES):", "\t" + response.text]
                print(log)
            if de_response:
                return {'state': 1, 'msg': '成功', 'body': json.loads(de_response)}
            else:
                return {'state': -1, 'msg': '解密失败', 'response': response.text}
        else:
            if is_debug:
                log += ["Receive data:", "", "Receive data(AES):", "\t" + response.text]
                print(log)
            return {'state': 0, 'msg': '通信失败', 'response': response.text}

    @staticmethod
    def encrypt(data: str, key: str) -> str:
        """
        加密处理
        :param data: 需要加密的字符串
        :param key: 密钥，只有前16位参与加密
        :return: 加密后的字符串，base64编码
        """
        # 初始化AES加密器，只取前16位作为密钥
        cipher = AES.new(key[:16].encode('utf-8'), AES.MODE_ECB)
        # 使用PKCS7填充数据
        padded_data = pad(data.encode('utf-8'), AES.block_size)
        # 加密填充后的数据
        encrypted_data = cipher.encrypt(padded_data)
        # 返回Base64编码的加密结果
        return base64.b64encode(encrypted_data).decode('utf-8')

    @staticmethod
    def decrypt(data: str, key: str) -> str:
        """
        解密
        :param data: 需要解密的字符串(base64编码)
        :param key: 密钥，只有前16位参与解密
        :return: 解密后的字符串
        """
        cipher = AES.new(key[:16].encode('utf-8'), AES.MODE_ECB)
        decrypted_data = cipher.decrypt(base64.b64decode(data))
        unpad = lambda s: s[:-ord(s[len(s) - 1:])]
        try:
            return unpad(decrypted_data).decode('utf-8')
        except:
            return ''

    @staticmethod
    def rand_str(i_length: int = 4) -> str:
        """
        生成随机字符串
        :param i_length: 字符串长度，默认4
        :return: 生成的随机字符串
        """
        chars = 'ABCDEFGHIJKLMNPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890'
        return ''.join(random.choice(chars) for _ in range(i_length))

    @staticmethod
    def json_format(data: str, indent: str = '    ') -> str:
        """
        Json数据格式化
        :param data: 需要格式化的json字符串
        :param indent: 缩进字符，默认4个空格
        :return: 格式化后的字符串
        """
        parsed = json.loads(data)
        return json.dumps(parsed, indent=indent, ensure_ascii=False)


if __name__ == '__main__':
    gc_api = GcOpenApi()
    timestamp = int(time.time())
    pkg = 'igc_base.example.template'
    cls = 'GET_USER_INFO'
    post_data = {'name': '王小明', 'age': 43}
    res = gc_api.exec_api(pkg, cls, post_data, is_debug=True)
    print(res)
